import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { API } from '../App';
import { toast } from 'sonner';
import {
  Wallet,
  Clock,
  CheckCircle,
  XCircle,
  LogOut,
  Upload,
  ExternalLink,
  Calendar,
  DollarSign,
  AlertCircle,
  Info
} from 'lucide-react';
import { format } from 'date-fns';
import { fr } from 'date-fns/locale';

const Dashboard = ({ user, onLogout }) => {
  const [balance, setBalance] = useState({ pending: 0, available: 0, withdrawn: 0 });
  const [tasks, setTasks] = useState([]);
  const [reservations, setReservations] = useState([]);
  const [withdrawals, setWithdrawals] = useState([]);
  const [loading, setLoading] = useState(true);
  const [paypalAccount, setPaypalAccount] = useState(user.paypal_account || '');
  const [withdrawalAmount, setWithdrawalAmount] = useState('');
  const [selectedReservation, setSelectedReservation] = useState(null);
  const [proofData, setProofData] = useState({ image: null, link: '' });

  useEffect(() => {
    loadData();
  }, []);

  const loadData = async () => {
    try {
      const [balanceRes, tasksRes, reservationsRes, withdrawalsRes] = await Promise.all([
        axios.get(`${API}/balance`),
        axios.get(`${API}/tasks`),
        axios.get(`${API}/tasks/my-reservations`),
        axios.get(`${API}/withdrawals/history`)
      ]);

      setBalance(balanceRes.data);
      setTasks(tasksRes.data);
      setReservations(reservationsRes.data);
      setWithdrawals(withdrawalsRes.data);
    } catch (error) {
      toast.error('Erreur lors du chargement des données');
    } finally {
      setLoading(false);
    }
  };

  const handleReserveTask = async (taskId) => {
    try {
      await axios.post(`${API}/tasks/reserve?task_id=${taskId}`);
      toast.success('Tâche réservée avec succès!');
      loadData();
    } catch (error) {
      toast.error(error.response?.data?.detail || 'Erreur lors de la réservation');
    }
  };

  const handleSubmitProof = async (e) => {
    e.preventDefault();
    if (!proofData.image || !proofData.link) {
      toast.error('Veuillez fournir une image et un lien');
      return;
    }

    const formData = new FormData();
    formData.append('reservation_id', selectedReservation.id);
    formData.append('review_link', proofData.link);
    formData.append('proof_image', proofData.image);

    try {
      await axios.post(`${API}/tasks/submit-proof`, formData, {
        headers: { 'Content-Type': 'multipart/form-data' }
      });
      toast.success('Preuve envoyée! En attente de validation');
      setSelectedReservation(null);
      setProofData({ image: null, link: '' });
      loadData();
    } catch (error) {
      toast.error(error.response?.data?.detail || 'Erreur lors de l\'envoi');
    }
  };

  const handleCancelReservation = async (reservationId) => {
    try {
      await axios.post(`${API}/tasks/cancel-reservation?reservation_id=${reservationId}`);
      toast.success('Réservation annulée');
      loadData();
    } catch (error) {
      toast.error(error.response?.data?.detail || 'Erreur lors de l\'annulation');
    }
  };

  const handleUpdatePaypal = async () => {
    try {
      await axios.put(`${API}/auth/profile?paypal_account=${encodeURIComponent(paypalAccount)}`);
      toast.success('Compte PayPal mis à jour');
    } catch (error) {
      toast.error('Erreur lors de la mise à jour');
    }
  };

  const handleRequestWithdrawal = async (e) => {
    e.preventDefault();
    if (!paypalAccount) {
      toast.error('Veuillez configurer votre compte PayPal d\'abord');
      return;
    }

    try {
      await axios.post(`${API}/withdrawals/request`, { amount: parseFloat(withdrawalAmount) });
      toast.success('Demande de retrait envoyée');
      setWithdrawalAmount('');
      loadData();
    } catch (error) {
      toast.error(error.response?.data?.detail || 'Erreur lors de la demande');
    }
  };

  const formatDate = (dateString) => {
    if (!dateString) return '-';
    return format(new Date(dateString), 'dd/MM/yyyy', { locale: fr });
  };

  const activeReservations = reservations.filter(r => r.status === 'reserved');
  const submittedReservations = reservations.filter(r => r.status === 'submitted');
  const validatedReservations = reservations.filter(r => ['validated', 'paid'].includes(r.status));
  const rejectedReservations = reservations.filter(r => r.status === 'rejected');

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-emerald-600"></div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-50" data-testid="user-dashboard">
      {/* Header */}
      <header className="bg-white border-b border-gray-200 shadow-sm">
        <div className="max-w-7xl mx-auto px-4 py-4 flex justify-between items-center">
          <div>
            <h1 className="text-2xl font-bold text-gray-900" style={{ fontFamily: 'Inter, sans-serif' }}>Genloss Avis</h1>
            <p className="text-sm text-gray-600">Bienvenue, {user.discord_pseudo}</p>
          </div>
          <button
            onClick={onLogout}
            className="flex items-center gap-2 px-4 py-2 text-gray-700 hover:text-gray-900 hover:bg-gray-100 rounded-lg"
            data-testid="logout-btn"
          >
            <LogOut className="w-4 h-4" />
            Déconnexion
          </button>
        </div>
      </header>

      <main className="max-w-7xl mx-auto px-4 py-8">
        {/* Balance Cards */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
          <div className="bg-gradient-to-br from-yellow-50 to-yellow-100 border border-yellow-200 rounded-xl p-6 shadow-sm" data-testid="pending-balance-card">
            <div className="flex items-center gap-3 mb-2">
              <Clock className="w-6 h-6 text-yellow-600" />
              <h3 className="font-semibold text-gray-900">Solde en attente</h3>
            </div>
            <p className="text-3xl font-bold text-yellow-700">€{balance.pending.toFixed(2)}</p>
            <p className="text-sm text-yellow-600 mt-1">En cours de validation</p>
          </div>

          <div className="bg-gradient-to-br from-emerald-50 to-emerald-100 border border-emerald-200 rounded-xl p-6 shadow-sm" data-testid="available-balance-card">
            <div className="flex items-center gap-3 mb-2">
              <Wallet className="w-6 h-6 text-emerald-600" />
              <h3 className="font-semibold text-gray-900">Solde disponible</h3>
            </div>
            <p className="text-3xl font-bold text-emerald-700">€{balance.available.toFixed(2)}</p>
            <p className="text-sm text-emerald-600 mt-1">Prêt à retirer</p>
          </div>

          <div className="bg-gradient-to-br from-gray-50 to-gray-100 border border-gray-200 rounded-xl p-6 shadow-sm" data-testid="withdrawn-balance-card">
            <div className="flex items-center gap-3 mb-2">
              <CheckCircle className="w-6 h-6 text-gray-600" />
              <h3 className="font-semibold text-gray-900">Solde retiré</h3>
            </div>
            <p className="text-3xl font-bold text-gray-700">€{balance.withdrawn.toFixed(2)}</p>
            <p className="text-sm text-gray-600 mt-1">Déjà payé</p>
          </div>
        </div>

        {/* PayPal Account */}
        <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6 mb-8" data-testid="paypal-section">
          <h2 className="text-xl font-bold text-gray-900 mb-4">Compte PayPal</h2>
          <div className="flex gap-3">
            <input
              type="email"
              value={paypalAccount}
              onChange={(e) => setPaypalAccount(e.target.value)}
              placeholder="votre@email-paypal.com"
              className="flex-1 px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-emerald-500 focus:border-transparent outline-none"
              data-testid="paypal-input"
            />
            <button
              onClick={handleUpdatePaypal}
              className="px-6 py-2 bg-emerald-600 text-white rounded-lg hover:bg-emerald-700 font-medium"
              data-testid="update-paypal-btn"
            >
              Mettre à jour
            </button>
          </div>
        </div>

        {/* Available Tasks */}
        <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6 mb-8" data-testid="available-tasks-section">
          <h2 className="text-xl font-bold text-gray-900 mb-4">Tâches disponibles</h2>
          {activeReservations.length > 0 && (
            <div className="mb-4 p-4 bg-blue-50 border border-blue-200 rounded-lg flex items-start gap-3" data-testid="active-reservation-notice">
              <Info className="w-5 h-5 text-blue-600 flex-shrink-0 mt-0.5" />
              <p className="text-sm text-blue-800">
                Vous avez déjà une réservation active. Veuillez la compléter avant d'en réserver une nouvelle.
              </p>
            </div>
          )}
          {tasks.length === 0 ? (
            <p className="text-gray-500 text-center py-8">Aucune tâche disponible pour le moment</p>
          ) : (
            <div className="grid gap-4">
              {tasks.map((task) => (
                <div key={task.id} className="border border-gray-200 rounded-lg p-4 hover:border-emerald-300 transition-colors" data-testid={`task-${task.id}`}>
                  <div className="flex justify-between items-start mb-3">
                    <div className="flex-1">
                      <h3 className="font-semibold text-gray-900 mb-1">{task.title}</h3>
                      <p className="text-sm text-gray-600 mb-2">{task.description}</p>
                      <a
                        href={task.target_link}
                        target="_blank"
                        rel="noopener noreferrer"
                        className="text-sm text-emerald-600 hover:text-emerald-700 inline-flex items-center gap-1"
                      >
                        <ExternalLink className="w-4 h-4" />
                        {task.target_link}
                      </a>
                    </div>
                    <div className="text-right ml-4">
                      <p className="text-2xl font-bold text-emerald-600">€{task.reward.toFixed(2)}</p>
                    </div>
                  </div>
                  <button
                    onClick={() => handleReserveTask(task.id)}
                    disabled={activeReservations.length > 0}
                    className="w-full py-2 bg-emerald-600 text-white rounded-lg hover:bg-emerald-700 disabled:opacity-50 disabled:cursor-not-allowed font-medium"
                    data-testid={`reserve-task-btn-${task.id}`}
                  >
                    Réserver cette tâche
                  </button>
                </div>
              ))}
            </div>
          )}
        </div>

        {/* Active Reservations */}
        {activeReservations.length > 0 && (
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6 mb-8" data-testid="active-reservations-section">
            <h2 className="text-xl font-bold text-gray-900 mb-4">Mes réservations actives</h2>
            <div className="space-y-4">
              {activeReservations.map((reservation) => (
                <div key={reservation.id} className="border border-gray-200 rounded-lg p-4" data-testid={`reservation-${reservation.id}`}>
                  {reservation.task && (
                    <>
                      <h3 className="font-semibold text-gray-900 mb-2">{reservation.task.title}</h3>
                      <p className="text-sm text-gray-600 mb-2">{reservation.task.description}</p>
                      <div className="text-sm text-gray-500 mb-3">
                        <Calendar className="inline w-4 h-4 mr-1" />
                        Réservé le: {formatDate(reservation.reserved_at)}
                      </div>

                      {!selectedReservation || selectedReservation.id !== reservation.id ? (
                        <div className="flex gap-2">
                          <button
                            onClick={() => setSelectedReservation(reservation)}
                            className="flex-1 py-2 bg-emerald-600 text-white rounded-lg hover:bg-emerald-700 font-medium flex items-center justify-center gap-2"
                            data-testid={`submit-proof-btn-${reservation.id}`}
                          >
                            <Upload className="w-4 h-4" />
                            Envoyer la preuve
                          </button>
                          <button
                            onClick={() => handleCancelReservation(reservation.id)}
                            className="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 font-medium"
                            data-testid={`cancel-reservation-btn-${reservation.id}`}
                          >
                            Annuler
                          </button>
                        </div>
                      ) : (
                        <form onSubmit={handleSubmitProof} className="space-y-3 mt-4 bg-gray-50 p-4 rounded-lg" data-testid="proof-submission-form">
                          <div>
                            <label className="block text-sm font-medium text-gray-700 mb-1">Photo de preuve</label>
                            <input
                              type="file"
                              accept="image/*"
                              onChange={(e) => setProofData({ ...proofData, image: e.target.files[0] })}
                              required
                              className="w-full text-sm"
                              data-testid="proof-image-input"
                            />
                          </div>
                          <div>
                            <label className="block text-sm font-medium text-gray-700 mb-1">Lien de la critique</label>
                            <input
                              type="url"
                              value={proofData.link}
                              onChange={(e) => setProofData({ ...proofData, link: e.target.value })}
                              required
                              placeholder="https://..."
                              className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-emerald-500 focus:border-transparent outline-none"
                              data-testid="proof-link-input"
                            />
                          </div>
                          <div className="flex gap-2">
                            <button
                              type="submit"
                              className="flex-1 py-2 bg-emerald-600 text-white rounded-lg hover:bg-emerald-700 font-medium"
                              data-testid="submit-proof-confirm-btn"
                            >
                              Valider
                            </button>
                            <button
                              type="button"
                              onClick={() => {
                                setSelectedReservation(null);
                                setProofData({ image: null, link: '' });
                              }}
                              className="px-4 py-2 bg-gray-500 text-white rounded-lg hover:bg-gray-600 font-medium"
                              data-testid="cancel-proof-btn"
                            >
                              Annuler
                            </button>
                          </div>
                        </form>
                      )}
                    </>
                  )}
                </div>
              ))}
            </div>
          </div>
        )}

        {/* Submitted (En validation) */}
        {submittedReservations.length > 0 && (
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6 mb-8" data-testid="submitted-reservations-section">
            <h2 className="text-xl font-bold text-gray-900 mb-4">En validation</h2>
            <div className="space-y-3">
              {submittedReservations.map((reservation) => (
                <div key={reservation.id} className="border border-yellow-200 bg-yellow-50 rounded-lg p-4" data-testid={`submitted-${reservation.id}`}>
                  {reservation.task && (
                    <>
                      <h3 className="font-semibold text-gray-900">{reservation.task.title}</h3>
                      <p className="text-sm text-gray-600 mt-1">Envoyé le: {formatDate(reservation.submitted_at)}</p>
                      <p className="text-sm text-yellow-700 mt-2 flex items-center gap-2">
                        <Clock className="w-4 h-4" />
                        En attente de validation par l'admin
                      </p>
                    </>
                  )}
                </div>
              ))}
            </div>
          </div>
        )}

        {/* Rejected */}
        {rejectedReservations.length > 0 && (
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6 mb-8" data-testid="rejected-reservations-section">
            <h2 className="text-xl font-bold text-gray-900 mb-4">Critiques rejetées</h2>
            <div className="space-y-3">
              {rejectedReservations.map((reservation) => (
                <div key={reservation.id} className="border border-red-200 bg-red-50 rounded-lg p-4" data-testid={`rejected-${reservation.id}`}>
                  {reservation.task && (
                    <>
                      <h3 className="font-semibold text-gray-900">{reservation.task.title}</h3>
                      {reservation.rejection_reason && (
                        <p className="text-sm text-red-700 mt-2 flex items-start gap-2">
                          <XCircle className="w-4 h-4 flex-shrink-0 mt-0.5" />
                          {reservation.rejection_reason}
                        </p>
                      )}
                    </>
                  )}
                </div>
              ))}
            </div>
          </div>
        )}

        {/* Validated/Paid */}
        {validatedReservations.length > 0 && (
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6 mb-8" data-testid="paid-reservations-section">
            <h2 className="text-xl font-bold text-gray-900 mb-4">Critiques payées</h2>
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead>
                  <tr className="border-b border-gray-200 text-left">
                    <th className="py-3 px-4 font-semibold text-gray-700">Tâche</th>
                    <th className="py-3 px-4 font-semibold text-gray-700">Montant</th>
                    <th className="py-3 px-4 font-semibold text-gray-700">Approuvé</th>
                    <th className="py-3 px-4 font-semibold text-gray-700">Payé</th>
                    <th className="py-3 px-4 font-semibold text-gray-700">Statut</th>
                  </tr>
                </thead>
                <tbody>
                  {validatedReservations.map((reservation) => (
                    <tr key={reservation.id} className="border-b border-gray-100" data-testid={`paid-${reservation.id}`}>
                      <td className="py-3 px-4">{reservation.task?.title || '-'}</td>
                      <td className="py-3 px-4 font-semibold text-emerald-600">€{reservation.task?.reward.toFixed(2)}</td>
                      <td className="py-3 px-4 text-sm text-gray-600">{formatDate(reservation.validated_at)}</td>
                      <td className="py-3 px-4 text-sm text-gray-600">{formatDate(reservation.paid_at)}</td>
                      <td className="py-3 px-4">
                        {reservation.status === 'paid' ? (
                          <span className="inline-flex items-center gap-1 px-2 py-1 bg-green-100 text-green-800 rounded text-xs font-medium">
                            <CheckCircle className="w-3 h-3" />
                            Payé
                          </span>
                        ) : (
                          <span className="inline-flex items-center gap-1 px-2 py-1 bg-yellow-100 text-yellow-800 rounded text-xs font-medium">
                            <Clock className="w-3 h-3" />
                            Validé
                          </span>
                        )}
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          </div>
        )}

        {/* Withdrawal Section */}
        <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6 mb-8" data-testid="withdrawal-section">
          <h2 className="text-xl font-bold text-gray-900 mb-4">Demander un retrait</h2>
          <form onSubmit={handleRequestWithdrawal} className="flex gap-3">
            <div className="flex-1">
              <input
                type="number"
                step="0.01"
                min="0.01"
                value={withdrawalAmount}
                onChange={(e) => setWithdrawalAmount(e.target.value)}
                placeholder="Montant à retirer"
                className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-emerald-500 focus:border-transparent outline-none"
                required
                data-testid="withdrawal-amount-input"
              />
              <p className="text-xs text-gray-500 mt-1">Solde disponible: €{balance.available.toFixed(2)}</p>
            </div>
            <button
              type="submit"
              className="px-6 py-2 bg-emerald-600 text-white rounded-lg hover:bg-emerald-700 font-medium flex items-center gap-2"
              data-testid="request-withdrawal-btn"
            >
              <DollarSign className="w-4 h-4" />
              Demander
            </button>
          </form>

          {/* Withdrawal History */}
          {withdrawals.length > 0 && (
            <div className="mt-6" data-testid="withdrawal-history">
              <h3 className="font-semibold text-gray-900 mb-3">Historique des retraits</h3>
              <div className="space-y-2">
                {withdrawals.map((withdrawal) => (
                  <div key={withdrawal.id} className="flex justify-between items-center p-3 bg-gray-50 rounded-lg" data-testid={`withdrawal-${withdrawal.id}`}>
                    <div>
                      <p className="font-medium text-gray-900">€{withdrawal.amount.toFixed(2)}</p>
                      <p className="text-xs text-gray-500">Demandé le: {formatDate(withdrawal.requested_at)}</p>
                    </div>
                    <div className="text-right">
                      {withdrawal.status === 'paid' && (
                        <span className="inline-flex items-center gap-1 px-3 py-1 bg-green-100 text-green-800 rounded-full text-xs font-medium">
                          <CheckCircle className="w-3 h-3" />
                          Payé le {formatDate(withdrawal.paid_at)}
                        </span>
                      )}
                      {withdrawal.status === 'approved' && (
                        <span className="inline-flex items-center gap-1 px-3 py-1 bg-blue-100 text-blue-800 rounded-full text-xs font-medium">
                          <CheckCircle className="w-3 h-3" />
                          Approuvé
                        </span>
                      )}
                      {withdrawal.status === 'pending' && (
                        <span className="inline-flex items-center gap-1 px-3 py-1 bg-yellow-100 text-yellow-800 rounded-full text-xs font-medium">
                          <Clock className="w-3 h-3" />
                          En attente
                        </span>
                      )}
                    </div>
                  </div>
                ))}
              </div>
            </div>
          )}
        </div>

        {/* Footer */}
        <div className="text-center text-sm text-gray-600 py-8" data-testid="dashboard-footer">
          <p>
            Un problème?{' '}
            <a
              href="https://discord.gg/yEarkSJhEB"
              target="_blank"
              rel="noopener noreferrer"
              className="text-emerald-600 hover:text-emerald-700 font-medium hover:underline"
            >
              Contactez-nous sur Discord
            </a>
          </p>
        </div>
      </main>
    </div>
  );
};

export default Dashboard;
