import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { API } from '../App';
import { toast } from 'sonner';
import {
  Users,
  ClipboardList,
  CheckCircle,
  XCircle,
  DollarSign,
  LogOut,
  Plus,
  Edit,
  Eye,
  Calendar,
  Mail
} from 'lucide-react';
import { format } from 'date-fns';
import { fr } from 'date-fns/locale';

const AdminPanel = ({ user, onLogout }) => {
  const [activeTab, setActiveTab] = useState('tasks');
  const [tasks, setTasks] = useState([]);
  const [validations, setValidations] = useState([]);
  const [users, setUsers] = useState([]);
  const [withdrawals, setWithdrawals] = useState([]);
  const [loading, setLoading] = useState(true);

  // Task form
  const [taskForm, setTaskForm] = useState({
    title: '',
    description: '',
    target_link: '',
    reward: ''
  });

  // User form
  const [userForm, setUserForm] = useState({
    email: '',
    discord_pseudo: '',
    password: '',
    role: 'user'
  });
  const [showUserForm, setShowUserForm] = useState(false);
  const [editingUser, setEditingUser] = useState(null);

  // View proof modal
  const [viewProof, setViewProof] = useState(null);

  useEffect(() => {
    loadData();
  }, [activeTab]);

  const loadData = async () => {
    setLoading(true);
    try {
      if (activeTab === 'tasks') {
        const res = await axios.get(`${API}/admin/tasks`);
        setTasks(res.data);
      } else if (activeTab === 'validations') {
        const res = await axios.get(`${API}/admin/validations`);
        setValidations(res.data);
      } else if (activeTab === 'users') {
        const res = await axios.get(`${API}/admin/users`);
        setUsers(res.data);
      } else if (activeTab === 'withdrawals') {
        const res = await axios.get(`${API}/admin/withdrawals`);
        setWithdrawals(res.data);
      }
    } catch (error) {
      toast.error('Erreur lors du chargement des données');
    } finally {
      setLoading(false);
    }
  };

  const handleCreateTask = async (e) => {
    e.preventDefault();
    try {
      await axios.post(`${API}/admin/tasks`, {
        ...taskForm,
        reward: parseFloat(taskForm.reward)
      });
      toast.success('Tâche créée avec succès');
      setTaskForm({ title: '', description: '', target_link: '', reward: '' });
      loadData();
    } catch (error) {
      toast.error(error.response?.data?.detail || 'Erreur lors de la création');
    }
  };

  const handleApproveValidation = async (reservationId) => {
    try {
      await axios.post(`${API}/admin/validations/approve?reservation_id=${reservationId}`);
      toast.success('Critique approuvée');
      loadData();
    } catch (error) {
      toast.error('Erreur lors de l\'approbation');
    }
  };

  const handleRejectValidation = async (reservationId) => {
    const reason = prompt('Raison du rejet:');
    if (!reason) return;

    try {
      await axios.post(`${API}/admin/validations/reject?reservation_id=${reservationId}&reason=${encodeURIComponent(reason)}`);
      toast.success('Critique rejetée');
      loadData();
    } catch (error) {
      toast.error('Erreur lors du rejet');
    }
  };

  const handleMarkAsPaid = async (reservationId) => {
    try {
      await axios.post(`${API}/admin/reservations/mark-paid?reservation_id=${reservationId}`);
      toast.success('Marqué comme payé');
      loadData();
    } catch (error) {
      toast.error('Erreur lors de la mise à jour');
    }
  };

  const handleCreateUser = async (e) => {
    e.preventDefault();
    try {
      await axios.post(`${API}/admin/users/create?role=${userForm.role}`, {
        email: userForm.email,
        discord_pseudo: userForm.discord_pseudo,
        password: userForm.password
      });
      toast.success('Utilisateur créé');
      setUserForm({ email: '', discord_pseudo: '', password: '', role: 'user' });
      setShowUserForm(false);
      loadData();
    } catch (error) {
      toast.error(error.response?.data?.detail || 'Erreur lors de la création');
    }
  };

  const handleUpdateUser = async (userId) => {
    const newPseudo = prompt('Nouveau pseudo Discord:', editingUser?.discord_pseudo);
    if (!newPseudo) return;

    try {
      await axios.put(`${API}/admin/users/${userId}`, {
        discord_pseudo: newPseudo
      });
      toast.success('Utilisateur mis à jour');
      setEditingUser(null);
      loadData();
    } catch (error) {
      toast.error('Erreur lors de la mise à jour');
    }
  };

  const handleToggleUserActive = async (userId, currentStatus) => {
    try {
      await axios.put(`${API}/admin/users/${userId}`, {
        is_active: !currentStatus
      });
      toast.success('Statut utilisateur mis à jour');
      loadData();
    } catch (error) {
      toast.error('Erreur lors de la mise à jour');
    }
  };

  const handleApproveWithdrawal = async (withdrawalId) => {
    try {
      await axios.post(`${API}/admin/withdrawals/approve?withdrawal_id=${withdrawalId}`);
      toast.success('Retrait approuvé');
      loadData();
    } catch (error) {
      toast.error('Erreur');
    }
  };

  const handleMarkWithdrawalPaid = async (withdrawalId) => {
    try {
      await axios.post(`${API}/admin/withdrawals/mark-paid?withdrawal_id=${withdrawalId}`);
      toast.success('Retrait marqué comme payé');
      loadData();
    } catch (error) {
      toast.error('Erreur');
    }
  };

  const formatDate = (dateString) => {
    if (!dateString) return '-';
    return format(new Date(dateString), 'dd/MM/yyyy HH:mm', { locale: fr });
  };

  return (
    <div className="min-h-screen bg-gray-50" data-testid="admin-panel">
      {/* Header */}
      <header className="bg-gradient-to-r from-emerald-600 to-emerald-700 text-white shadow-lg">
        <div className="max-w-7xl mx-auto px-4 py-4 flex justify-between items-center">
          <div>
            <h1 className="text-2xl font-bold" style={{ fontFamily: 'Inter, sans-serif' }}>Panel Admin - Genloss</h1>
            <p className="text-sm text-emerald-100">Connecté en tant que {user.email}</p>
          </div>
          <button
            onClick={onLogout}
            className="flex items-center gap-2 px-4 py-2 bg-white/10 hover:bg-white/20 rounded-lg transition-colors"
            data-testid="admin-logout-btn"
          >
            <LogOut className="w-4 h-4" />
            Déconnexion
          </button>
        </div>
      </header>

      {/* Tabs */}
      <div className="bg-white border-b border-gray-200 shadow-sm" data-testid="admin-tabs">
        <div className="max-w-7xl mx-auto px-4">
          <div className="flex gap-1">
            {[
              { id: 'tasks', label: 'Tâches', icon: ClipboardList },
              { id: 'validations', label: 'En validation', icon: CheckCircle },
              { id: 'users', label: 'Utilisateurs', icon: Users },
              { id: 'withdrawals', label: 'Retraits', icon: DollarSign }
            ].map((tab) => (
              <button
                key={tab.id}
                onClick={() => setActiveTab(tab.id)}
                className={`flex items-center gap-2 px-6 py-4 font-medium border-b-2 transition-colors ${
                  activeTab === tab.id
                    ? 'border-emerald-600 text-emerald-600'
                    : 'border-transparent text-gray-600 hover:text-gray-900'
                }`}
                data-testid={`tab-${tab.id}`}
              >
                <tab.icon className="w-4 h-4" />
                {tab.label}
              </button>
            ))}
          </div>
        </div>
      </div>

      <main className="max-w-7xl mx-auto px-4 py-8">
        {loading ? (
          <div className="flex items-center justify-center py-12">
            <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-emerald-600"></div>
          </div>
        ) : (
          <>
            {/* Tasks Tab */}
            {activeTab === 'tasks' && (
              <div data-testid="tasks-tab">
                <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6 mb-6">
                  <h2 className="text-xl font-bold text-gray-900 mb-4">Créer une nouvelle tâche</h2>
                  <form onSubmit={handleCreateTask} className="space-y-4" data-testid="create-task-form">
                    <input
                      type="text"
                      value={taskForm.title}
                      onChange={(e) => setTaskForm({ ...taskForm, title: e.target.value })}
                      placeholder="Titre de la tâche"
                      required
                      className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-emerald-500 focus:border-transparent outline-none"
                      data-testid="task-title-input"
                    />
                    <textarea
                      value={taskForm.description}
                      onChange={(e) => setTaskForm({ ...taskForm, description: e.target.value })}
                      placeholder="Description détaillée"
                      required
                      rows="3"
                      className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-emerald-500 focus:border-transparent outline-none"
                      data-testid="task-description-input"
                    />
                    <input
                      type="url"
                      value={taskForm.target_link}
                      onChange={(e) => setTaskForm({ ...taskForm, target_link: e.target.value })}
                      placeholder="Lien cible (ex: https://...)"
                      required
                      className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-emerald-500 focus:border-transparent outline-none"
                      data-testid="task-link-input"
                    />
                    <input
                      type="number"
                      step="0.01"
                      min="0.01"
                      value={taskForm.reward}
                      onChange={(e) => setTaskForm({ ...taskForm, reward: e.target.value })}
                      placeholder="Rémunération (€)"
                      required
                      className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-emerald-500 focus:border-transparent outline-none"
                      data-testid="task-reward-input"
                    />
                    <button
                      type="submit"
                      className="w-full py-3 bg-emerald-600 text-white rounded-lg hover:bg-emerald-700 font-medium flex items-center justify-center gap-2"
                      data-testid="create-task-btn"
                    >
                      <Plus className="w-4 h-4" />
                      Créer la tâche
                    </button>
                  </form>
                </div>

                <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
                  <h2 className="text-xl font-bold text-gray-900 mb-4">Toutes les tâches ({tasks.length})</h2>
                  <div className="space-y-3">
                    {tasks.map((task) => (
                      <div key={task.id} className="border border-gray-200 rounded-lg p-4" data-testid={`admin-task-${task.id}`}>
                        <h3 className="font-semibold text-gray-900">{task.title}</h3>
                        <p className="text-sm text-gray-600 mt-1">{task.description}</p>
                        <div className="flex justify-between items-center mt-3">
                          <span className="text-lg font-bold text-emerald-600">€{task.reward.toFixed(2)}</span>
                          <span className="text-xs text-gray-500">Créé le: {formatDate(task.created_at)}</span>
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              </div>
            )}

            {/* Validations Tab */}
            {activeTab === 'validations' && (
              <div data-testid="validations-tab">
                <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
                  <h2 className="text-xl font-bold text-gray-900 mb-4">Critiques en attente de validation ({validations.length})</h2>
                  {validations.length === 0 ? (
                    <p className="text-gray-500 text-center py-8">Aucune critique en attente</p>
                  ) : (
                    <div className="space-y-4">
                      {validations.map((item) => (
                        <div key={item.reservation.id} className="border border-gray-200 rounded-lg p-4" data-testid={`validation-${item.reservation.id}`}>
                          <div className="flex justify-between items-start mb-3">
                            <div className="flex-1">
                              <h3 className="font-semibold text-gray-900">{item.task?.title}</h3>
                              <p className="text-sm text-gray-600 mt-1">
                                <Mail className="inline w-4 h-4 mr-1" />
                                Utilisateur: {item.user?.discord_pseudo} ({item.user?.email})
                              </p>
                              <p className="text-sm text-gray-500 mt-1">
                                <Calendar className="inline w-4 h-4 mr-1" />
                                Envoyé le: {formatDate(item.reservation.submitted_at)}
                              </p>
                            </div>
                            <span className="text-xl font-bold text-emerald-600">€{item.task?.reward.toFixed(2)}</span>
                          </div>

                          <div className="bg-gray-50 rounded-lg p-3 mb-3">
                            <p className="text-sm text-gray-700 mb-2">
                              <strong>Lien de la critique:</strong>{' '}
                              <a
                                href={item.reservation.review_link}
                                target="_blank"
                                rel="noopener noreferrer"
                                className="text-emerald-600 hover:underline"
                              >
                                {item.reservation.review_link}
                              </a>
                            </p>
                            <button
                              onClick={() => setViewProof(item.reservation)}
                              className="text-sm text-emerald-600 hover:text-emerald-700 flex items-center gap-1"
                              data-testid={`view-proof-btn-${item.reservation.id}`}
                            >
                              <Eye className="w-4 h-4" />
                              Voir la preuve (image)
                            </button>
                          </div>

                          <div className="flex gap-2">
                            <button
                              onClick={() => handleApproveValidation(item.reservation.id)}
                              className="flex-1 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 font-medium flex items-center justify-center gap-2"
                              data-testid={`approve-btn-${item.reservation.id}`}
                            >
                              <CheckCircle className="w-4 h-4" />
                              Approuver
                            </button>
                            <button
                              onClick={() => handleRejectValidation(item.reservation.id)}
                              className="flex-1 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 font-medium flex items-center justify-center gap-2"
                              data-testid={`reject-btn-${item.reservation.id}`}
                            >
                              <XCircle className="w-4 h-4" />
                              Rejeter
                            </button>
                          </div>
                        </div>
                      ))}
                    </div>
                  )}
                </div>
              </div>
            )}

            {/* Users Tab */}
            {activeTab === 'users' && (
              <div data-testid="users-tab">
                <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6 mb-6">
                  <div className="flex justify-between items-center mb-4">
                    <h2 className="text-xl font-bold text-gray-900">Créer un utilisateur</h2>
                    <button
                      onClick={() => setShowUserForm(!showUserForm)}
                      className="px-4 py-2 bg-emerald-600 text-white rounded-lg hover:bg-emerald-700 font-medium flex items-center gap-2"
                      data-testid="toggle-user-form-btn"
                    >
                      <Plus className="w-4 h-4" />
                      {showUserForm ? 'Annuler' : 'Nouveau'}
                    </button>
                  </div>

                  {showUserForm && (
                    <form onSubmit={handleCreateUser} className="space-y-4" data-testid="create-user-form">
                      <input
                        type="email"
                        value={userForm.email}
                        onChange={(e) => setUserForm({ ...userForm, email: e.target.value })}
                        placeholder="Email"
                        required
                        className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-emerald-500 focus:border-transparent outline-none"
                        data-testid="user-email-input"
                      />
                      <input
                        type="text"
                        value={userForm.discord_pseudo}
                        onChange={(e) => setUserForm({ ...userForm, discord_pseudo: e.target.value })}
                        placeholder="Pseudo Discord"
                        required
                        className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-emerald-500 focus:border-transparent outline-none"
                        data-testid="user-pseudo-input"
                      />
                      <input
                        type="password"
                        value={userForm.password}
                        onChange={(e) => setUserForm({ ...userForm, password: e.target.value })}
                        placeholder="Mot de passe"
                        required
                        className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-emerald-500 focus:border-transparent outline-none"
                        data-testid="user-password-input"
                      />
                      <select
                        value={userForm.role}
                        onChange={(e) => setUserForm({ ...userForm, role: e.target.value })}
                        className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-emerald-500 focus:border-transparent outline-none"
                        data-testid="user-role-select"
                      >
                        <option value="user">Utilisateur</option>
                        <option value="admin">Admin</option>
                      </select>
                      <button
                        type="submit"
                        className="w-full py-3 bg-emerald-600 text-white rounded-lg hover:bg-emerald-700 font-medium"
                        data-testid="create-user-btn"
                      >
                        Créer l'utilisateur
                      </button>
                    </form>
                  )}
                </div>

                <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
                  <h2 className="text-xl font-bold text-gray-900 mb-4">Tous les utilisateurs ({users.length})</h2>
                  <div className="overflow-x-auto">
                    <table className="w-full">
                      <thead>
                        <tr className="border-b border-gray-200 text-left">
                          <th className="py-3 px-4 font-semibold text-gray-700">Email</th>
                          <th className="py-3 px-4 font-semibold text-gray-700">Discord</th>
                          <th className="py-3 px-4 font-semibold text-gray-700">PayPal</th>
                          <th className="py-3 px-4 font-semibold text-gray-700">Rôle</th>
                          <th className="py-3 px-4 font-semibold text-gray-700">Statut</th>
                          <th className="py-3 px-4 font-semibold text-gray-700">Actions</th>
                        </tr>
                      </thead>
                      <tbody>
                        {users.map((u) => (
                          <tr key={u.id} className="border-b border-gray-100" data-testid={`user-${u.id}`}>
                            <td className="py-3 px-4">{u.email}</td>
                            <td className="py-3 px-4">{u.discord_pseudo}</td>
                            <td className="py-3 px-4 text-sm text-gray-600">{u.paypal_account || '-'}</td>
                            <td className="py-3 px-4">
                              <span className={`px-2 py-1 rounded text-xs font-medium ${
                                u.role === 'admin' ? 'bg-purple-100 text-purple-800' : 'bg-gray-100 text-gray-800'
                              }`}>
                                {u.role}
                              </span>
                            </td>
                            <td className="py-3 px-4">
                              <button
                                onClick={() => handleToggleUserActive(u.id, u.is_active)}
                                className={`px-3 py-1 rounded-full text-xs font-medium ${
                                  u.is_active ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'
                                }`}
                                data-testid={`toggle-active-${u.id}`}
                              >
                                {u.is_active ? 'Actif' : 'Inactif'}
                              </button>
                            </td>
                            <td className="py-3 px-4">
                              <button
                                onClick={() => {
                                  setEditingUser(u);
                                  handleUpdateUser(u.id);
                                }}
                                className="text-emerald-600 hover:text-emerald-700"
                                data-testid={`edit-user-${u.id}`}
                              >
                                <Edit className="w-4 h-4" />
                              </button>
                            </td>
                          </tr>
                        ))}
                      </tbody>
                    </table>
                  </div>
                </div>
              </div>
            )}

            {/* Withdrawals Tab */}
            {activeTab === 'withdrawals' && (
              <div data-testid="withdrawals-tab">
                <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
                  <h2 className="text-xl font-bold text-gray-900 mb-4">Demandes de retrait ({withdrawals.length})</h2>
                  {withdrawals.length === 0 ? (
                    <p className="text-gray-500 text-center py-8">Aucune demande de retrait</p>
                  ) : (
                    <div className="overflow-x-auto">
                      <table className="w-full">
                        <thead>
                          <tr className="border-b border-gray-200 text-left">
                            <th className="py-3 px-4 font-semibold text-gray-700">Utilisateur</th>
                            <th className="py-3 px-4 font-semibold text-gray-700">PayPal</th>
                            <th className="py-3 px-4 font-semibold text-gray-700">Montant</th>
                            <th className="py-3 px-4 font-semibold text-gray-700">Date demande</th>
                            <th className="py-3 px-4 font-semibold text-gray-700">Statut</th>
                            <th className="py-3 px-4 font-semibold text-gray-700">Actions</th>
                          </tr>
                        </thead>
                        <tbody>
                          {withdrawals.map((item) => (
                            <tr key={item.withdrawal.id} className="border-b border-gray-100" data-testid={`withdrawal-${item.withdrawal.id}`}>
                              <td className="py-3 px-4">
                                <div>
                                  <p className="font-medium text-gray-900">{item.user?.discord_pseudo}</p>
                                  <p className="text-xs text-gray-500">{item.user?.email}</p>
                                </div>
                              </td>
                              <td className="py-3 px-4 text-sm">{item.withdrawal.paypal_account}</td>
                              <td className="py-3 px-4 font-bold text-emerald-600">€{item.withdrawal.amount.toFixed(2)}</td>
                              <td className="py-3 px-4 text-sm text-gray-600">{formatDate(item.withdrawal.requested_at)}</td>
                              <td className="py-3 px-4">
                                {item.withdrawal.status === 'paid' && (
                                  <span className="px-2 py-1 bg-green-100 text-green-800 rounded text-xs font-medium">
                                    Payé
                                  </span>
                                )}
                                {item.withdrawal.status === 'approved' && (
                                  <span className="px-2 py-1 bg-blue-100 text-blue-800 rounded text-xs font-medium">
                                    Approuvé
                                  </span>
                                )}
                                {item.withdrawal.status === 'pending' && (
                                  <span className="px-2 py-1 bg-yellow-100 text-yellow-800 rounded text-xs font-medium">
                                    En attente
                                  </span>
                                )}
                              </td>
                              <td className="py-3 px-4">
                                <div className="flex gap-2">
                                  {item.withdrawal.status === 'pending' && (
                                    <button
                                      onClick={() => handleApproveWithdrawal(item.withdrawal.id)}
                                      className="px-3 py-1 bg-blue-600 text-white rounded text-sm hover:bg-blue-700"
                                      data-testid={`approve-withdrawal-${item.withdrawal.id}`}
                                    >
                                      Approuver
                                    </button>
                                  )}
                                  {item.withdrawal.status === 'approved' && (
                                    <button
                                      onClick={() => handleMarkWithdrawalPaid(item.withdrawal.id)}
                                      className="px-3 py-1 bg-green-600 text-white rounded text-sm hover:bg-green-700"
                                      data-testid={`mark-paid-${item.withdrawal.id}`}
                                    >
                                      Marquer payé
                                    </button>
                                  )}
                                </div>
                              </td>
                            </tr>
                          ))}
                        </tbody>
                      </table>
                    </div>
                  )}
                </div>
              </div>
            )}
          </>
        )}
      </main>

      {/* View Proof Modal */}
      {viewProof && (
        <div
          className="fixed inset-0 bg-black/50 flex items-center justify-center z-50 p-4"
          onClick={() => setViewProof(null)}
          data-testid="proof-modal"
        >
          <div
            className="bg-white rounded-xl max-w-2xl w-full p-6"
            onClick={(e) => e.stopPropagation()}
          >
            <h3 className="text-xl font-bold text-gray-900 mb-4">Preuve de la critique</h3>
            <img
              src={`${process.env.REACT_APP_BACKEND_URL}/uploads/${viewProof.proof_image}`}
              alt="Preuve"
              className="w-full rounded-lg border border-gray-200"
            />
            <button
              onClick={() => setViewProof(null)}
              className="mt-4 w-full py-2 bg-gray-600 text-white rounded-lg hover:bg-gray-700"
              data-testid="close-proof-modal-btn"
            >
              Fermer
            </button>
          </div>
        </div>
      )}

      {/* Footer */}
      <div className="text-center text-sm text-gray-600 py-8">
        <p>
          Un problème?{' '}
          <a
            href="https://discord.gg/yEarkSJhEB"
            target="_blank"
            rel="noopener noreferrer"
            className="text-emerald-600 hover:text-emerald-700 font-medium hover:underline"
          >
            Contactez-nous sur Discord
          </a>
        </p>
      </div>
    </div>
  );
};

export default AdminPanel;
