from fastapi import FastAPI, APIRouter, HTTPException, Depends, UploadFile, File, Form, status
from fastapi.security import HTTPBearer, HTTPAuthorizationCredentials
from fastapi.staticfiles import StaticFiles
from dotenv import load_dotenv
from starlette.middleware.cors import CORSMiddleware
from motor.motor_asyncio import AsyncIOMotorClient
import os
import logging
from pathlib import Path
from pydantic import BaseModel, Field, ConfigDict, EmailStr
from typing import List, Optional
import uuid
from datetime import datetime, timezone, timedelta
import bcrypt
import jwt
import shutil

ROOT_DIR = Path(__file__).parent
load_dotenv(ROOT_DIR / '.env')

# MongoDB connection
mongo_url = os.environ['MONGO_URL']
client = AsyncIOMotorClient(mongo_url)
db = client[os.environ['DB_NAME']]

# JWT Configuration
JWT_SECRET = os.environ.get('JWT_SECRET', 'your-secret-key-change-in-production')
JWT_ALGORITHM = 'HS256'
JWT_EXPIRATION_DAYS = 30

# Create uploads directory
UPLOADS_DIR = ROOT_DIR / 'uploads'
UPLOADS_DIR.mkdir(exist_ok=True)

# Security
security = HTTPBearer()

# Create the main app
app = FastAPI()

# Create a router with the /api prefix
api_router = APIRouter(prefix="/api")

# ============= MODELS =============

class UserBase(BaseModel):
    email: EmailStr
    discord_pseudo: str
    
class UserCreate(UserBase):
    password: str

class UserLogin(BaseModel):
    email: EmailStr
    password: str

class User(UserBase):
    model_config = ConfigDict(extra="ignore")
    id: str
    role: str = "user"  # user or admin
    paypal_account: Optional[str] = None
    is_active: bool = True
    created_at: str

class UserUpdate(BaseModel):
    discord_pseudo: Optional[str] = None
    password: Optional[str] = None
    paypal_account: Optional[str] = None
    is_active: Optional[bool] = None
    role: Optional[str] = None

class Task(BaseModel):
    model_config = ConfigDict(extra="ignore")
    id: str
    title: str
    description: str
    target_link: str
    reward: float
    created_by: str
    created_at: str
    is_available: bool = True

class TaskCreate(BaseModel):
    title: str
    description: str
    target_link: str
    reward: float

class Reservation(BaseModel):
    model_config = ConfigDict(extra="ignore")
    id: str
    user_id: str
    task_id: str
    status: str  # reserved, submitted, validated, rejected, paid
    proof_image: Optional[str] = None
    review_link: Optional[str] = None
    reserved_at: str
    submitted_at: Optional[str] = None
    validated_at: Optional[str] = None
    paid_at: Optional[str] = None
    rejection_reason: Optional[str] = None

class ReservationWithTask(Reservation):
    task: Optional[Task] = None

class Withdrawal(BaseModel):
    model_config = ConfigDict(extra="ignore")
    id: str
    user_id: str
    amount: float
    paypal_account: str
    status: str  # pending, approved, paid
    requested_at: str
    approved_at: Optional[str] = None
    paid_at: Optional[str] = None

class WithdrawalRequest(BaseModel):
    amount: float

class Balance(BaseModel):
    pending: float
    available: float
    withdrawn: float

class TokenResponse(BaseModel):
    access_token: str
    token_type: str = "bearer"
    user: User

# ============= AUTH HELPERS =============

def hash_password(password: str) -> str:
    return bcrypt.hashpw(password.encode('utf-8'), bcrypt.gensalt()).decode('utf-8')

def verify_password(password: str, hashed: str) -> bool:
    return bcrypt.checkpw(password.encode('utf-8'), hashed.encode('utf-8'))

def create_token(user_id: str) -> str:
    expiration = datetime.now(timezone.utc) + timedelta(days=JWT_EXPIRATION_DAYS)
    payload = {
        'user_id': user_id,
        'exp': expiration
    }
    return jwt.encode(payload, JWT_SECRET, algorithm=JWT_ALGORITHM)

async def get_current_user(credentials: HTTPAuthorizationCredentials = Depends(security)) -> User:
    try:
        token = credentials.credentials
        payload = jwt.decode(token, JWT_SECRET, algorithms=[JWT_ALGORITHM])
        user_id = payload['user_id']
        
        user_doc = await db.users.find_one({"id": user_id}, {"_id": 0})
        if not user_doc or not user_doc.get('is_active', True):
            raise HTTPException(status_code=401, detail="Utilisateur non trouvé ou inactif")
        
        return User(**user_doc)
    except jwt.ExpiredSignatureError:
        raise HTTPException(status_code=401, detail="Token expiré")
    except Exception as e:
        raise HTTPException(status_code=401, detail="Token invalide")

async def require_admin(user: User = Depends(get_current_user)) -> User:
    if user.role != "admin":
        raise HTTPException(status_code=403, detail="Accès admin requis")
    return user

# ============= AUTH ENDPOINTS =============

@api_router.post("/auth/register", response_model=TokenResponse)
async def register(user_data: UserCreate):
    # Check if user exists
    existing = await db.users.find_one({"email": user_data.email}, {"_id": 0})
    if existing:
        raise HTTPException(status_code=400, detail="Cet email est déjà utilisé")
    
    # Create user
    user_id = str(uuid.uuid4())
    user_doc = {
        "id": user_id,
        "email": user_data.email,
        "discord_pseudo": user_data.discord_pseudo,
        "password": hash_password(user_data.password),
        "role": "user",
        "paypal_account": None,
        "is_active": True,
        "created_at": datetime.now(timezone.utc).isoformat()
    }
    
    await db.users.insert_one(user_doc)
    
    user_doc.pop('password')
    token = create_token(user_id)
    
    return TokenResponse(
        access_token=token,
        user=User(**user_doc)
    )

@api_router.post("/auth/login", response_model=TokenResponse)
async def login(login_data: UserLogin):
    user_doc = await db.users.find_one({"email": login_data.email}, {"_id": 0})
    
    if not user_doc or not verify_password(login_data.password, user_doc['password']):
        raise HTTPException(status_code=401, detail="Email ou mot de passe incorrect")
    
    if not user_doc.get('is_active', True):
        raise HTTPException(status_code=403, detail="Compte désactivé")
    
    user_doc.pop('password')
    token = create_token(user_doc['id'])
    
    return TokenResponse(
        access_token=token,
        user=User(**user_doc)
    )

@api_router.get("/auth/me", response_model=User)
async def get_me(user: User = Depends(get_current_user)):
    return user

@api_router.put("/auth/profile")
async def update_profile(paypal_account: str, user: User = Depends(get_current_user)):
    await db.users.update_one(
        {"id": user.id},
        {"$set": {"paypal_account": paypal_account}}
    )
    return {"message": "Profil mis à jour"}

# ============= TASK ENDPOINTS =============

@api_router.get("/tasks", response_model=List[Task])
async def get_available_tasks(user: User = Depends(get_current_user)):
    # Get tasks not reserved by anyone
    tasks = await db.tasks.find({"is_available": True}, {"_id": 0}).to_list(1000)
    
    # Check which tasks are reserved
    for task in tasks:
        reservation = await db.reservations.find_one({
            "task_id": task['id'],
            "status": {"$in": ["reserved", "submitted"]}
        })
        if reservation:
            task['is_available'] = False
    
    return [Task(**task) for task in tasks if task['is_available']]

@api_router.post("/tasks/reserve")
async def reserve_task(task_id: str, user: User = Depends(get_current_user)):
    # Check if user already has an active reservation
    existing = await db.reservations.find_one({
        "user_id": user.id,
        "status": {"$in": ["reserved", "submitted"]}
    })
    
    if existing:
        raise HTTPException(status_code=400, detail="Vous avez déjà une réservation en cours")
    
    # Check if task exists and is available
    task = await db.tasks.find_one({"id": task_id}, {"_id": 0})
    if not task:
        raise HTTPException(status_code=404, detail="Tâche non trouvée")
    
    # Check if task is already reserved
    reserved = await db.reservations.find_one({
        "task_id": task_id,
        "status": {"$in": ["reserved", "submitted"]}
    })
    
    if reserved:
        raise HTTPException(status_code=400, detail="Cette tâche est déjà réservée")
    
    # Create reservation
    reservation_id = str(uuid.uuid4())
    reservation_doc = {
        "id": reservation_id,
        "user_id": user.id,
        "task_id": task_id,
        "status": "reserved",
        "reserved_at": datetime.now(timezone.utc).isoformat()
    }
    
    await db.reservations.insert_one(reservation_doc)
    
    return {"message": "Tâche réservée avec succès", "reservation_id": reservation_id}

@api_router.get("/tasks/my-reservations", response_model=List[ReservationWithTask])
async def get_my_reservations(user: User = Depends(get_current_user)):
    reservations = await db.reservations.find({"user_id": user.id}, {"_id": 0}).to_list(1000)
    
    result = []
    for res in reservations:
        task = await db.tasks.find_one({"id": res['task_id']}, {"_id": 0})
        res_with_task = ReservationWithTask(**res)
        if task:
            res_with_task.task = Task(**task)
        result.append(res_with_task)
    
    return result

@api_router.post("/tasks/submit-proof")
async def submit_proof(
    reservation_id: str = Form(...),
    review_link: str = Form(...),
    proof_image: UploadFile = File(...),
    user: User = Depends(get_current_user)
):
    # Find reservation
    reservation = await db.reservations.find_one(
        {"id": reservation_id, "user_id": user.id},
        {"_id": 0}
    )
    
    if not reservation:
        raise HTTPException(status_code=404, detail="Réservation non trouvée")
    
    if reservation['status'] != "reserved":
        raise HTTPException(status_code=400, detail="Cette réservation n'est pas en attente de preuve")
    
    # Save image
    file_extension = proof_image.filename.split('.')[-1]
    filename = f"{reservation_id}_{uuid.uuid4()}.{file_extension}"
    file_path = UPLOADS_DIR / filename
    
    with file_path.open('wb') as buffer:
        shutil.copyfileobj(proof_image.file, buffer)
    
    # Update reservation
    await db.reservations.update_one(
        {"id": reservation_id},
        {"$set": {
            "status": "submitted",
            "proof_image": filename,
            "review_link": review_link,
            "submitted_at": datetime.now(timezone.utc).isoformat()
        }}
    )
    
    return {"message": "Preuve envoyée avec succès - En attente de validation"}

@api_router.post("/tasks/cancel-reservation")
async def cancel_reservation(reservation_id: str, user: User = Depends(get_current_user)):
    reservation = await db.reservations.find_one(
        {"id": reservation_id, "user_id": user.id},
        {"_id": 0}
    )
    
    if not reservation:
        raise HTTPException(status_code=404, detail="Réservation non trouvée")
    
    if reservation['status'] not in ["reserved"]:
        raise HTTPException(status_code=400, detail="Cette réservation ne peut pas être annulée")
    
    await db.reservations.delete_one({"id": reservation_id})
    
    return {"message": "Réservation annulée"}

# ============= BALANCE ENDPOINTS =============

@api_router.get("/balance", response_model=Balance)
async def get_balance(user: User = Depends(get_current_user)):
    # Get validated reservations
    validated = await db.reservations.find({
        "user_id": user.id,
        "status": {"$in": ["validated", "paid"]}
    }, {"_id": 0}).to_list(1000)
    
    validated_tasks = []
    for res in validated:
        task = await db.tasks.find_one({"id": res['task_id']}, {"_id": 0})
        if task:
            validated_tasks.append({"reservation": res, "task": task})
    
    # Calculate pending (validated but not paid)
    pending = sum(
        item['task']['reward']
        for item in validated_tasks
        if item['reservation']['status'] == 'validated'
    )
    
    # Calculate total earned (all validated + paid)
    total_earned = sum(item['task']['reward'] for item in validated_tasks)
    
    # Get withdrawals
    withdrawals = await db.withdrawals.find({
        "user_id": user.id,
        "status": "paid"
    }, {"_id": 0}).to_list(1000)
    
    withdrawn = sum(w['amount'] for w in withdrawals)
    
    available = total_earned - withdrawn
    
    return Balance(
        pending=pending,
        available=available,
        withdrawn=withdrawn
    )

# ============= WITHDRAWAL ENDPOINTS =============

@api_router.post("/withdrawals/request")
async def request_withdrawal(
    withdrawal_data: WithdrawalRequest,
    user: User = Depends(get_current_user)
):
    if not user.paypal_account:
        raise HTTPException(status_code=400, detail="Veuillez configurer votre compte PayPal dans votre profil")
    
    # Check balance
    balance = await get_balance(user)
    if withdrawal_data.amount > balance.available:
        raise HTTPException(status_code=400, detail="Solde insuffisant")
    
    if withdrawal_data.amount <= 0:
        raise HTTPException(status_code=400, detail="Montant invalide")
    
    # Create withdrawal request
    withdrawal_id = str(uuid.uuid4())
    withdrawal_doc = {
        "id": withdrawal_id,
        "user_id": user.id,
        "amount": withdrawal_data.amount,
        "paypal_account": user.paypal_account,
        "status": "pending",
        "requested_at": datetime.now(timezone.utc).isoformat()
    }
    
    await db.withdrawals.insert_one(withdrawal_doc)
    
    return {"message": "Demande de retrait envoyée", "withdrawal_id": withdrawal_id}

@api_router.get("/withdrawals/history", response_model=List[Withdrawal])
async def get_withdrawal_history(user: User = Depends(get_current_user)):
    withdrawals = await db.withdrawals.find({"user_id": user.id}, {"_id": 0}).to_list(1000)
    return [Withdrawal(**w) for w in withdrawals]

# ============= ADMIN ENDPOINTS =============

@api_router.post("/admin/tasks", response_model=Task)
async def create_task(task_data: TaskCreate, admin: User = Depends(require_admin)):
    task_id = str(uuid.uuid4())
    task_doc = {
        "id": task_id,
        "title": task_data.title,
        "description": task_data.description,
        "target_link": task_data.target_link,
        "reward": task_data.reward,
        "created_by": admin.id,
        "created_at": datetime.now(timezone.utc).isoformat(),
        "is_available": True
    }
    
    await db.tasks.insert_one(task_doc)
    
    return Task(**task_doc)

@api_router.get("/admin/tasks", response_model=List[Task])
async def get_all_tasks(admin: User = Depends(require_admin)):
    tasks = await db.tasks.find({}, {"_id": 0}).to_list(1000)
    return [Task(**task) for task in tasks]

@api_router.get("/admin/validations")
async def get_pending_validations(admin: User = Depends(require_admin)):
    reservations = await db.reservations.find(
        {"status": "submitted"},
        {"_id": 0}
    ).to_list(1000)
    
    result = []
    for res in reservations:
        task = await db.tasks.find_one({"id": res['task_id']}, {"_id": 0})
        user = await db.users.find_one({"id": res['user_id']}, {"_id": 0, "password": 0})
        result.append({
            "reservation": res,
            "task": task,
            "user": user
        })
    
    return result

@api_router.post("/admin/validations/approve")
async def approve_validation(reservation_id: str, admin: User = Depends(require_admin)):
    reservation = await db.reservations.find_one({"id": reservation_id}, {"_id": 0})
    
    if not reservation:
        raise HTTPException(status_code=404, detail="Réservation non trouvée")
    
    await db.reservations.update_one(
        {"id": reservation_id},
        {"$set": {
            "status": "validated",
            "validated_at": datetime.now(timezone.utc).isoformat()
        }}
    )
    
    return {"message": "Critique approuvée"}

@api_router.post("/admin/validations/reject")
async def reject_validation(reservation_id: str, reason: str, admin: User = Depends(require_admin)):
    reservation = await db.reservations.find_one({"id": reservation_id}, {"_id": 0})
    
    if not reservation:
        raise HTTPException(status_code=404, detail="Réservation non trouvée")
    
    await db.reservations.update_one(
        {"id": reservation_id},
        {"$set": {
            "status": "rejected",
            "rejection_reason": reason
        }}
    )
    
    return {"message": "Critique rejetée"}

@api_router.post("/admin/reservations/mark-paid")
async def mark_as_paid(reservation_id: str, admin: User = Depends(require_admin)):
    reservation = await db.reservations.find_one({"id": reservation_id}, {"_id": 0})
    
    if not reservation:
        raise HTTPException(status_code=404, detail="Réservation non trouvée")
    
    if reservation['status'] != 'validated':
        raise HTTPException(status_code=400, detail="La réservation doit être validée d'abord")
    
    await db.reservations.update_one(
        {"id": reservation_id},
        {"$set": {
            "status": "paid",
            "paid_at": datetime.now(timezone.utc).isoformat()
        }}
    )
    
    return {"message": "Marqué comme payé"}

@api_router.get("/admin/users")
async def get_all_users(admin: User = Depends(require_admin)):
    users = await db.users.find({}, {"_id": 0, "password": 0}).to_list(1000)
    return users

@api_router.put("/admin/users/{user_id}")
async def update_user(user_id: str, update_data: UserUpdate, admin: User = Depends(require_admin)):
    update_dict = {k: v for k, v in update_data.model_dump().items() if v is not None}
    
    if 'password' in update_dict:
        update_dict['password'] = hash_password(update_dict['password'])
    
    await db.users.update_one(
        {"id": user_id},
        {"$set": update_dict}
    )
    
    return {"message": "Utilisateur mis à jour"}

@api_router.post("/admin/users/create")
async def create_user_admin(user_data: UserCreate, role: str = "user", admin: User = Depends(require_admin)):
    # Check if user exists
    existing = await db.users.find_one({"email": user_data.email}, {"_id": 0})
    if existing:
        raise HTTPException(status_code=400, detail="Cet email est déjà utilisé")
    
    # Create user
    user_id = str(uuid.uuid4())
    user_doc = {
        "id": user_id,
        "email": user_data.email,
        "discord_pseudo": user_data.discord_pseudo,
        "password": hash_password(user_data.password),
        "role": role,
        "paypal_account": None,
        "is_active": True,
        "created_at": datetime.now(timezone.utc).isoformat()
    }
    
    await db.users.insert_one(user_doc)
    
    return {"message": "Utilisateur créé", "user_id": user_id}

@api_router.get("/admin/withdrawals")
async def get_all_withdrawals(admin: User = Depends(require_admin)):
    withdrawals = await db.withdrawals.find({}, {"_id": 0}).to_list(1000)
    
    result = []
    for w in withdrawals:
        user = await db.users.find_one({"id": w['user_id']}, {"_id": 0, "password": 0})
        result.append({
            "withdrawal": w,
            "user": user
        })
    
    return result

@api_router.post("/admin/withdrawals/approve")
async def approve_withdrawal(withdrawal_id: str, admin: User = Depends(require_admin)):
    await db.withdrawals.update_one(
        {"id": withdrawal_id},
        {"$set": {
            "status": "approved",
            "approved_at": datetime.now(timezone.utc).isoformat()
        }}
    )
    
    return {"message": "Retrait approuvé"}

@api_router.post("/admin/withdrawals/mark-paid")
async def mark_withdrawal_paid(withdrawal_id: str, admin: User = Depends(require_admin)):
    withdrawal = await db.withdrawals.find_one({"id": withdrawal_id}, {"_id": 0})
    
    if not withdrawal:
        raise HTTPException(status_code=404, detail="Retrait non trouvé")
    
    await db.withdrawals.update_one(
        {"id": withdrawal_id},
        {"$set": {
            "status": "paid",
            "paid_at": datetime.now(timezone.utc).isoformat()
        }}
    )
    
    return {"message": "Retrait marqué comme payé"}

# Mount uploads directory
app.mount("/uploads", StaticFiles(directory=str(UPLOADS_DIR)), name="uploads")

# Include the router in the main app
app.include_router(api_router)

app.add_middleware(
    CORSMiddleware,
    allow_credentials=True,
    allow_origins=os.environ.get('CORS_ORIGINS', '*').split(','),
    allow_methods=["*"],
    allow_headers=["*"],
)

# Configure logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)

@app.on_event("shutdown")
async def shutdown_db_client():
    client.close()