import asyncio
from motor.motor_asyncio import AsyncIOMotorClient
import os
from dotenv import load_dotenv
from pathlib import Path
import bcrypt
import uuid
from datetime import datetime, timezone

ROOT_DIR = Path(__file__).parent
load_dotenv(ROOT_DIR / '.env')

mongo_url = os.environ['MONGO_URL']
client = AsyncIOMotorClient(mongo_url)
db = client[os.environ['DB_NAME']]

def hash_password(password: str) -> str:
    return bcrypt.hashpw(password.encode('utf-8'), bcrypt.gensalt()).decode('utf-8')

async def seed_database():
    print("🌱 Seeding database...")
    
    # Clear existing data
    await db.users.delete_many({})
    await db.tasks.delete_many({})
    await db.reservations.delete_many({})
    await db.withdrawals.delete_many({})
    
    # Create admin user
    admin_id = str(uuid.uuid4())
    admin = {
        "id": admin_id,
        "email": "ephacks6@gmail.com",
        "discord_pseudo": "AdminGenloss",
        "password": hash_password("admin123"),
        "role": "admin",
        "paypal_account": "ephacks6@gmail.com",
        "is_active": True,
        "created_at": datetime.now(timezone.utc).isoformat()
    }
    await db.users.insert_one(admin)
    print(f"✅ Admin créé: {admin['email']} / mot de passe: admin123")
    
    # Create test user
    user_id = str(uuid.uuid4())
    user = {
        "id": user_id,
        "email": "test@example.com",
        "discord_pseudo": "TestUser#1234",
        "password": hash_password("test123"),
        "role": "user",
        "paypal_account": "testuser@paypal.com",
        "is_active": True,
        "created_at": datetime.now(timezone.utc).isoformat()
    }
    await db.users.insert_one(user)
    print(f"✅ Utilisateur test créé: {user['email']} / mot de passe: test123")
    
    # Create sample tasks
    tasks = [
        {
            "id": str(uuid.uuid4()),
            "title": "Rédiger un avis sur notre produit X",
            "description": "Rédigez un avis détaillé et honnête sur notre produit X. L'avis doit faire au moins 100 mots et mentionner vos points forts préférés.",
            "target_link": "https://example.com/product-x",
            "reward": 5.0,
            "created_by": admin_id,
            "created_at": datetime.now(timezone.utc).isoformat(),
            "is_available": True
        },
        {
            "id": str(uuid.uuid4()),
            "title": "Avis Google My Business",
            "description": "Laissez un avis 5 étoiles sur notre page Google My Business. Partagez votre expérience avec notre service.",
            "target_link": "https://g.page/example",
            "reward": 8.0,
            "created_by": admin_id,
            "created_at": datetime.now(timezone.utc).isoformat(),
            "is_available": True
        },
        {
            "id": str(uuid.uuid4()),
            "title": "Avis Trustpilot",
            "description": "Rédigez un avis complet sur notre page Trustpilot. Décrivez votre expérience d'achat et de service client.",
            "target_link": "https://www.trustpilot.com/review/example.com",
            "reward": 10.0,
            "created_by": admin_id,
            "created_at": datetime.now(timezone.utc).isoformat(),
            "is_available": True
        },
        {
            "id": str(uuid.uuid4()),
            "title": "Avis sur notre application mobile",
            "description": "Téléchargez notre app et laissez un avis sur l'App Store ou Google Play. N'oubliez pas de mettre 5 étoiles!",
            "target_link": "https://apps.apple.com/app/example",
            "reward": 7.0,
            "created_by": admin_id,
            "created_at": datetime.now(timezone.utc).isoformat(),
            "is_available": True
        }
    ]
    
    await db.tasks.insert_many(tasks)
    print(f"✅ {len(tasks)} tâches créées")
    
    print("\n🎉 Database seeded successfully!")
    print("\n📋 IDENTIFIANTS DE CONNEXION:")
    print("="*50)
    print("ADMIN:")
    print(f"  Email: ephacks6@gmail.com")
    print(f"  Mot de passe: admin123")
    print("\nUSER TEST:")
    print(f"  Email: test@example.com")
    print(f"  Mot de passe: test123")
    print("="*50)

if __name__ == "__main__":
    asyncio.run(seed_database())